#include "ransac.h"

// The generic RANSAC algorithm (http://en.wikipedia.org/wiki/RANSAC - see pseudocode)
bool CRansacLinear::getBestModel(LinearModel& model)
{
    int foundModel = false;
    int iterations = 0;

    while (iterations < mIterations) {

        QList<QPointF>  maybeInliers = getMaybeInliers();
        QList<QPointF>  consensusSet = maybeInliers;
        model                        = getModel(maybeInliers[0], maybeInliers[1]);

        for(int i = 0; i <  mObservationSet.size(); i++) {

            if( !isMember(mObservationSet[i], maybeInliers )) {

                if( fitsModel(mObservationSet[i], model )) {
                    consensusSet.append(mObservationSet[i]);
                }
            }
        }

        if(consensusSet.size() >= mRequiredInliers) {

            model            = getModel(consensusSet);
            qreal thisError  = getModelError(consensusSet, model);

            if(thisError < mBestError) {

                dump(consensusSet);

                mBestConsensusSet = consensusSet;
                mBestModel        = model;
                mBestError        = thisError;
                foundModel        = true;
            }
        }
        iterations++;
    }
    return foundModel;
}

// Get two random points from observation-set
QList<QPointF> CRansacLinear::getMaybeInliers() const
{
    QList<QPointF> maybeInliers;

    qreal listSize = mObservationSet.size();
    int randPointIndex_0 = (int)(listSize *  qrand()/(RAND_MAX - 1.0));
    int randPointIndex_1 = (int)(listSize *  qrand()/(RAND_MAX - 1.0));

    maybeInliers.append(mObservationSet[randPointIndex_0]);
    maybeInliers.append(mObservationSet[randPointIndex_1]);

    return maybeInliers;
}

// Get  linear equation parameters (slope and intercept) from two points
LinearModel CRansacLinear::getModel(const QPointF& p0, const QPointF& p1) const
{
    LinearModel model;

    model.mSlope     = (p1.y()-p0.y())/(p1.x()-p0.x());
    model.mIntercept = p0.y() - model.mSlope * p0.x();

    return model;
}

// Get linear equation parameters (slope and intercept) from observation data
// Use opencv to solve least-square equation (...i know, this can be done in many ways)
LinearModel CRansacLinear::getModel(const QList<QPointF>& observation) const
{
    LinearModel model;

    unsigned int noDataPoints = observation.size();

    CvMat* ptValMat = cvCreateMat(noDataPoints, 2, CV_64FC1);
    CvMat* ptVecMat = cvCreateMat(noDataPoints, 1, CV_64FC1);
    CvMat* ptDest   = cvCreateMat(2,            1, CV_64FC1);

    for(unsigned int i=0; i < noDataPoints; i++) {
        cvmSet(ptValMat, i, 0, observation[i].x());
        cvmSet(ptValMat, i, 1, 1.0);
        cvmSet(ptVecMat, i, 0, observation[i].y());
    }

    cvSolve(ptValMat, ptVecMat, ptDest, CV_SVD);

    model.mIntercept = cvmGet(ptDest, 1, 0);
    model.mSlope     = cvmGet(ptDest, 0, 0);

    cvReleaseMat(&ptValMat);
    cvReleaseMat(&ptVecMat);
    cvReleaseMat(&ptDest);

    return model;
}

// Check, whether point is a member of list
bool CRansacLinear::isMember(const QPointF& p, const QList<QPointF>& pList) const
{
    bool isMember = false;

    for(int i = 0; i < pList.size(); i++) {
        if(p == pList[i]) {
            isMember = true;
        }
    }
    return isMember;
}

// Get distance between point and model
qreal CRansacLinear::getDistance(const QPointF& p, const LinearModel& model) const
{
    qreal distance = INT_MAX;
    
    qreal x1 = -INT_MAX;
    QPointF p1 = QPointF(x1, x1*model.mSlope+model.mIntercept);

    qreal x2 = INT_MAX;
    QPointF p2 = QPointF(x2, x2*model.mSlope+model.mIntercept);

    // Parameterform
    QPointF sv = p1;
    QPointF rv = p2 -p1;

    QPointF psv  = p - sv;
    qreal rvpsv  = rv.x()*psv.x() + rv.y()*psv.y();
    qreal rvrv   = rv.x()*rv.x() + rv.y()*rv.y();

    if(rvrv) {
        qreal lambda = rvpsv / rvrv;

        QPointF f = sv + lambda * rv;
        QPointF c = p -f;

        distance = sqrt(pow(c.x(), 2)+pow(c.y(), 2));
    }

    return distance;
}

// Check, whether distance of point from model is smaller than a given treshold
bool CRansacLinear::fitsModel(const QPointF& p, const LinearModel& model) const
{
    bool fits = false;

    qreal distance = getDistance(p, model);

    if(distance < mTreshold) {
        fits = true;
    }

    return fits;
}

// Model error ~ maximum points fitting the model
int CRansacLinear::getModelError(const QList<QPointF>& pointList, const LinearModel& model) const
{
    int max_error = INT_MAX;

    QList<QPointF> consensusSet;

    for(int i = 0; i < pointList.size(); i++) {

        if(fitsModel(pointList[i], model)) {
            consensusSet.append(pointList[i]);
        }
    }
    return (max_error-consensusSet.size());
}

// Helper function to generate data-files (gnuplot...)
void CRansacLinear::dump(const QList<QPointF>& pointListF)
{
    QFile file;
    QString strData;

    static int file_index = 0;

    file.setFileName(QString("%1.dat").arg(file_index));
    file.open(QIODevice::WriteOnly);

    for(int i=0; i < pointListF.size(); i++) {
        qreal X = pointListF.at(i).x();
        qreal Y = pointListF.at(i).y();
        strData = QString("%1").arg(X, 0, 'f', 2) + " " + QString("%1").arg(Y, 0, 'f', 2)  + "\n";
        file.write(strData.toUtf8().data());
    }

    file_index++;
    file.close();
}
