#include <QtCore/QCoreApplication>
#include "ransac.h"

// Model-Parameters: y = m * x + b
const qreal MAX_OBSERVATION_POINTS = 1000.0;
const qreal OBSERVATION_STEPWIDTH  = 30.0;
const qreal b = 500.0;
const qreal m = ((qreal)(2<<15) -b)/ (qreal)MAX_OBSERVATION_POINTS;

// Using the polar form of the Box-Muller transformation to generate gaussian-distributed numbers
// From http://www.taygeta.com/random/gaussian.html
qreal gauss_rand() {
        qreal x1, x2, w;

        do {
                x1 = 2.0 * qrand()/RAND_MAX - 1.0;
                x2 = 2.0 * qrand()/RAND_MAX - 1.0;
                w = x1 * x1 + x2 * x2;
        } while (w >= 1.0);

        w = sqrt((-2.0 * log(w)) / w);
        return x1 * w;
}

// create a linear 16bit-observation with noise (y=m*x+b+noise)
QList<QPointF> createObservationSet()
{
    QList<QPointF> observation;

    for(int i = 0; i < (int)(MAX_OBSERVATION_POINTS/OBSERVATION_STEPWIDTH); i++) {

        for(int j = 0; j < (int)OBSERVATION_STEPWIDTH; j++) {

            qreal gauss_noise = gauss_rand();

            observation.append(QPointF((qreal)(i*OBSERVATION_STEPWIDTH),
                                       (qreal) OBSERVATION_STEPWIDTH * i * m + 10.0 * b * gauss_noise));
        }
    }
    return observation;
}

//main program
int main()
{
    LinearModel model = LinearModel(m, b);
    CRansacLinear Ransac;

    Ransac.setObservationSet(createObservationSet());
    Ransac.setModel(model);
    Ransac.setIterations(MAX_OBSERVATION_POINTS);
    Ransac.setRequiredInliers(2);
    Ransac.setTreshold(16.0);
    Ransac.getBestModel(model);
}
